import pandas as pd
import numpy as np
import os
from ast import literal_eval

O1_events = [
        ["GW150914", "G190047", "HL", "Prod1"],
        ["GW151012", "G197392", "HL", "Prod1"],
        ["GW151226", "G211117", "HL", "Prod2"],

]
O2_events = [
        ["GW170104", "G268556", "HL", "Prod1"],
        ["GW170608", "G288732", "HL", "Prod5"],
        ["GW170729", "G299775", "HL", "Prod1"],
        ["GW170809", "G296853", "HL", "Prod2"],
        ["GW170814", "G297595", "HLV", "Prod1"],
        ["GW170818", "G298171", "LV", "Prod2"],
        ["GW170823", "G298936", "HL", "Prod1"]

]
O3a_events = [
        ["S190408an", "S190408an", "HLV", "Prod1"],
        ["S190412m", "S190412m", "HLV", "Prod1"],
        ["S190421ar", "S190421ar", "HL", "Prod1"],
        ["S190425z", "S190425z", "LV", "Prod2"],
        ["S190503bf", "S190503bf", "HLV", "Prod1"],
        ["S190512at", "S190512at", "HL", "Prod1"],
        ["S190513bm", "S190513bm", "HL", "Prod1"],
        ["S190517h", "S190517h", "HLV", "Prod1"],
        ["S190519bj", "S190519bj", "HLV", "Prod1"],
        ["S190521g", "S190521g", "HLV", "Prod2"],
        ["S190521r", "S190521r", "HL", "Prod1"],
        ["S190602aq", "S190602aq", "HLV", "Prod1"],
        ["S190630ag", "S190630ag", "LV", "Prod1"],
        ["S190706ai", "S190706ai", "HLV", "Prod2"],
        ["S190707q", "S190707q", "HL", "Prod3"],
        ["S190708ap", "S190708ap", "LV", "Prod1"],
        ["S190720a", "S190720a", "HLV", "Prod2"],
        ["S190727h", "S190727h", "HLV", "Prod3"],
        ["S190728q", "S190728q", "HLV", "Prod2"],
        ["S190814bv", "S190814bv", "LV", "Prod8"],
        ["S190828j", "S190828j", "HLV", "Prod1"],
        ["S190828l", "S190828l", "HLV", "Prod2"],
        ["S190915ak", "S190915ak", "HLV", "Prod1"],
        ["S190924h", "S190924h", "HLV", "Prod7"]

]
O3b_events = [
        ["S191109d", "S191109d", "HL", "Prod1"],
        ["S191129u", "S191129u", "HL", "Prod7"],
        ["S191204r", "S191204r", "HL", "Prod3"],
        ["S191215w", "S191215w", "HLV", "Prod1"],
        ["S191216ap", "S191216ap", "HV", "Prod6"],
        ["S191222n", "S191222n", "HL", "Prod1"],
        ["S200115j", "S200115j", "HLV", "Prod4"],
        ["S200129m", "S200129m", "HLV", "Prod1"],
        ["S200202ac", "S200202ac", "HLV", "Prod4"],
        ["S200208q", "S200208q", "HLV", "Prod3"],
        ["S200219ac", "S200219ac", "HLV", "Prod1"],
        ["S200224ca", "S200224ca", "HLV", "Prod1"],
        ["S200225q", "S200225q", "HL", "Prod1"],
        ["S200311bg", "S200311bg", "HLV", "Prod1"],
        ["S200316bj", "S200316bj", "HLV", "Prod2"]

]

def group_summary(label, events):
    """
    Group the result of the events into a summary table.

    ## Parameters
    -------------
    label: str
        Label of the events.
    events: array-like
        A matrix of name, dirname, ifos, label of the production run of the events.

    ## Returns
    ----------
    dfs_2: DataFrame
        DataFrame of the 2-detector events.
    dfs_3: DataFrame
        DataFrame of the 3-detector events.
    """
    dfs_2 = []
    dfs_3 = []

    for name, dirname, ifos, prod in events:
        fname = f"./{name}/summary.txt"
        if os.path.isfile(fname):
            df = pd.read_csv(fname)
            ifos = literal_eval(df["ifos"].iloc[0])
            nifo = len(ifos)
            df["ifos"] = [", ".join(ifos)]
            df["name"] = [name]
            if nifo == 2:
                df = df[["name", "ifos", "gpstime",
                         'log10B-1-bplus-cross',
                         'log10B-err-1-bplus-cross',
                         'log10B-1-bvectorx-vectory',
                         'log10B-err-1-bvectorx-vectory',
                         'log10B-1-breathing',
                         'log10B-err-1-breathing',
                         'log10B-1-bplus-cross-breathing-longitudinal',
                         'log10B-err-1-bplus-cross-breathing-longitudinal',
                         'log10B-1-bvectorx-vectory-breathing-longitudinal',
                         'log10B-err-1-bvectorx-vectory-breathing-longitudinal',
                         'log10B-1-bplus-cross-vectorx-vectory',
                         'log10B-err-1-bplus-cross-vectorx-vectory',
                         'log10B-1-bplus-cross-breathing-longitudinal-vectorx-vectory',
                         'log10B-err-1-bplus-cross-breathing-longitudinal-vectorx-vectory',]]
                dfs_2.append(df)
            elif nifo == 3:
                df = df[["name", "ifos", "gpstime",
                         'log10B-1-bplus-cross',
                         'log10B-err-1-bplus-cross',
                         'log10B-1-bvectorx-vectory',
                         'log10B-err-1-bvectorx-vectory',
                         'log10B-1-breathing',
                         'log10B-err-1-breathing',
                         'log10B-1-bplus-cross-breathing-longitudinal',
                         'log10B-err-1-bplus-cross-breathing-longitudinal',
                         'log10B-1-bvectorx-vectory-breathing-longitudinal',
                         'log10B-err-1-bvectorx-vectory-breathing-longitudinal',
                         'log10B-1-bplus-cross-vectorx-vectory',
                         'log10B-err-1-bplus-cross-vectorx-vectory',
                         'log10B-1-bplus-cross-breathing-longitudinal-vectorx-vectory',
                         'log10B-err-1-bplus-cross-breathing-longitudinal-vectorx-vectory',
                         'log10B-2-bplus-bcross',
                         'log10B-err-2-bplus-bcross',
                         'log10B-2-bvectorx-bvectory',
                         'log10B-err-2-bvectorx-bvectory',
                         'log10B-2-bplus-cross-bbreathing-longitudinal',
                         'log10B-err-2-bplus-cross-bbreathing-longitudinal',
                         'log10B-2-bvectorx-vectory-bbreathing-longitudinal',
                         'log10B-err-2-bvectorx-vectory-bbreathing-longitudinal',
                         'log10B-2-bplus-cross-bvectorx-vectory',
                         'log10B-err-2-bplus-cross-bvectorx-vectory',
                         'log10B-2-bplus-cross-bbreathing-longitudinal-vectorx-vectory',
                         'log10B-err-2-bplus-cross-bbreathing-longitudinal-vectorx-vectory',]]
                dfs_3.append(df)
        else:
            nifo = len(ifos)
            df = pd.DataFrame()
            df["name"] = [name]
            if nifo == 2:
                dfs_2.append(df)
            else:
                dfs_3.append(df)
            print(f"{fname} does not exist.")
    return dfs_2, dfs_3

def compute_combined_logB_l_1(df):
    """
    Calculate the combined log10 Bayes factor of the L = 2 analysis.

    Parameters
    ----------
    df: DataFrame
        A DataFrame of the results.

    Returns
    -------
    log10B, log10B_err
    """
    # Calculate the combined logB.
    clogB_S = df["log10B-1-breathing"].sum()
    clogB_S_err = np.sqrt(np.sum(df["log10B-err-1-breathing"]**2)) * 1.6449
    clogB_V = df["log10B-1-bvectorx-vectory"].sum()
    clogB_V_err = np.sqrt(np.sum(df["log10B-err-1-bvectorx-vectory"]**2)) * 1.6449
    clogB_TS = df["log10B-1-bplus-cross-breathing-longitudinal"].sum()
    clogB_TS_err = np.sqrt(np.sum(df["log10B-err-1-bplus-cross-breathing-longitudinal"]**2)) * 1.6449
    clogB_TV = df["log10B-1-bplus-cross-vectorx-vectory"].sum()
    clogB_TV_err = np.sqrt(np.sum(df["log10B-err-1-bplus-cross-vectorx-vectory"]**2)) * 1.6449
    clogB_VS = df["log10B-1-bvectorx-vectory-breathing-longitudinal"].sum()
    clogB_VS_err = np.sqrt(np.sum(df["log10B-err-1-bvectorx-vectory-breathing-longitudinal"]**2)) * 1.6449
    clogB_TVS = df["log10B-1-bplus-cross-breathing-longitudinal-vectorx-vectory"].sum()
    clogB_TVS_err = np.sqrt(np.sum(df["log10B-err-1-bplus-cross-breathing-longitudinal-vectorx-vectory"]**2)) * 1.6449

    return clogB_S, clogB_S_err, clogB_V, clogB_V_err, clogB_TS, clogB_TS_err, clogB_TV, clogB_TV_err, clogB_VS, clogB_VS_err, clogB_TVS, clogB_TVS_err

def compute_combined_logB_l_2(df):
    """
    Calculate the combined log10 Bayes factor of the L = 2 analysis.

    Parameters
    ----------
    df: DataFrame
        A DataFrame of the results.

    Returns
    -------
    log10B, log10B_err
    """
    # Calculate the combined logB.
    clogB_V = df["log10B-2-bvectorx-bvectory"].sum()
    clogB_V_err = np.sqrt(np.sum(df["log10B-err-2-bvectorx-bvectory"]**2)) * 1.6449
    clogB_TS = df["log10B-2-bplus-cross-bbreathing-longitudinal"].sum()
    clogB_TS_err = np.sqrt(np.sum(df["log10B-err-2-bplus-cross-bbreathing-longitudinal"]**2)) * 1.6449
    clogB_TV = df["log10B-2-bplus-cross-bvectorx-vectory"].sum()
    clogB_TV_err = np.sqrt(np.sum(df["log10B-err-2-bplus-cross-bvectorx-vectory"]**2)) * 1.6449
    clogB_VS = df["log10B-2-bvectorx-vectory-bbreathing-longitudinal"].sum()
    clogB_VS_err = np.sqrt(np.sum(df["log10B-err-2-bvectorx-vectory-bbreathing-longitudinal"]**2)) * 1.6449
    clogB_TVS = df["log10B-2-bplus-cross-bbreathing-longitudinal-vectorx-vectory"].sum()
    clogB_TVS_err = np.sqrt(np.sum(df["log10B-err-2-bplus-cross-bbreathing-longitudinal-vectorx-vectory"]**2)) * 1.6449

    return clogB_V, clogB_V_err, clogB_TS, clogB_TS_err, clogB_TV, clogB_TV_err, clogB_VS, clogB_VS_err, clogB_TVS, clogB_TVS_err



dfs_2_O1, dfs_3_O1 = group_summary("O1O2", O1_events)
dfs_2_O2, dfs_3_O2 = group_summary("O1O2", O2_events)
dfs_2_O3a, dfs_3_O3a = group_summary("O3a", O3a_events)
dfs_2_O3b, dfs_3_O3b = group_summary("O3b", O3b_events)

# Merge the DataFrames

df_l_1_O1 = pd.concat(dfs_2_O1 + dfs_3_O1)
df_l_1_O2 = pd.concat(dfs_2_O2 + dfs_3_O2)
df_l_1_O3a = pd.concat(dfs_2_O3a + dfs_3_O3a)
df_l_1_O3b = pd.concat(dfs_2_O3b + dfs_3_O3b)

#df_l_2_O1 = pd.concat(dfs_3_O1)
df_l_2_O2 = pd.concat(dfs_3_O2)
df_l_2_O3a = pd.concat(dfs_3_O3a)
df_l_2_O3b = pd.concat(dfs_3_O3b)

df_l_1_all = pd.concat(dfs_2_O1 + dfs_3_O1 +
                       dfs_2_O2 + dfs_3_O2 +
                       dfs_2_O3a + dfs_3_O3a +
                       dfs_2_O3b + dfs_3_O3b)

df_l_2_all = pd.concat(dfs_3_O1 +
                       dfs_3_O2 +
                       dfs_3_O3a +
                       dfs_3_O3b)

# Compute the combined log10 Bayes factor of the O1 events for the L = 1 analysis.
clogB_S_O1_1, clogB_S_err_O1_1, clogB_V_O1_1, clogB_V_err_O1_1, clogB_TS_O1_1, clogB_TS_err_O1_1, clogB_TV_O1_1, clogB_TV_err_O1_1, clogB_VS_O1_1, clogB_VS_err_O1_1, clogB_TVS_O1_1, clogB_TVS_err_O1_1 = compute_combined_logB_l_1(df_l_1_O1)

# Compute the combined log10 Bayes factor of the O2 events for the L = 1 analysis.
clogB_S_O2_1, clogB_S_err_O2_1, clogB_V_O2_1, clogB_V_err_O2_1, clogB_TS_O2_1, clogB_TS_err_O2_1, clogB_TV_O2_1, clogB_TV_err_O2_1, clogB_VS_O2_1, clogB_VS_err_O2_1, clogB_TVS_O2_1, clogB_TVS_err_O2_1 = compute_combined_logB_l_1(df_l_1_O2)

# Compute the combined log10 Bayes factor of the O3a events for the L = 1 analysis.
clogB_S_O3a_1, clogB_S_err_O3a_1, clogB_V_O3a_1, clogB_V_err_O3a_1, clogB_TS_O3a_1, clogB_TS_err_O3a_1, clogB_TV_O3a_1, clogB_TV_err_O3a_1, clogB_VS_O3a_1, clogB_VS_err_O3a_1, clogB_TVS_O3a_1, clogB_TVS_err_O3a_1 = compute_combined_logB_l_1(df_l_1_O3a)

# Compute the combined log10 Bayes factor of the O3b events for the L = 1 analysis.
clogB_S_O3b_1, clogB_S_err_O3b_1, clogB_V_O3b_1, clogB_V_err_O3b_1, clogB_TS_O3b_1, clogB_TS_err_O3b_1, clogB_TV_O3b_1, clogB_TV_err_O3b_1, clogB_VS_O3b_1, clogB_VS_err_O3b_1, clogB_TVS_O3b_1, clogB_TVS_err_O3b_1 = compute_combined_logB_l_1(df_l_1_O3b)

# Compute the combined log10 Bayes factor of all events for the L = 1 analysis.
clogB_S_all_1, clogB_S_err_all_1, clogB_V_all_1, clogB_V_err_all_1, clogB_TS_all_1, clogB_TS_err_all_1, clogB_TV_all_1, clogB_TV_err_all_1, clogB_VS_all_1, clogB_VS_err_all_1, clogB_TVS_all_1, clogB_TVS_err_all_1 = compute_combined_logB_l_1(df_l_1_all)

############################################################
# Write the combined results of the L = 1 analysis to file.
############################################################
with open("pol_table_1.tex", "w") as f:
    f.write("\\begin{table}\n")
    f.write("\t\\resizebox{\\textwidth}{!}{\n")
    f.write("\t\\begin{tabular}{c  c  c  c  c  c  c}\n")
    #f.write("\t\t\hline\n")
    f.write("\\toprule\n")
    f.write("\t\tEvents & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{S}}$ & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{V}}$ & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{TS}}$ & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{TV}}$ & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{VS}}$ & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{TVS}}$\\\\\n")
    #f.write("\t\t\hline\hline\n")
    f.write("\midrule\n")
    f.write("\t\tO1 & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ \\\\\n"%(
        clogB_S_O1_1,
        clogB_S_err_O1_1,
        clogB_V_O1_1,
        clogB_V_err_O1_1,
        clogB_TS_O1_1,
        clogB_TS_err_O1_1,
        clogB_TV_O1_1,
        clogB_TV_err_O1_1,
        clogB_VS_O1_1,
        clogB_VS_err_O1_1,
        clogB_TVS_O1_1,
        clogB_TVS_err_O1_1,
    ))
    f.write("\t\tO2 & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ \\\\\n"%(
        clogB_S_O2_1,
        clogB_S_err_O2_1,
        clogB_V_O2_1,
        clogB_V_err_O2_1,
        clogB_TS_O2_1,
        clogB_TS_err_O2_1,
        clogB_TV_O2_1,
        clogB_TV_err_O2_1,
        clogB_VS_O2_1,
        clogB_VS_err_O2_1,
        clogB_TVS_O2_1,
        clogB_TVS_err_O2_1,
    ))
    f.write("\t\tO3a & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ \\\\\n"%(
        clogB_S_O3a_1,
        clogB_S_err_O3a_1,
        clogB_V_O3a_1,
        clogB_V_err_O3a_1,
        clogB_TS_O3a_1,
        clogB_TS_err_O3a_1,
        clogB_TV_O3a_1,
        clogB_TV_err_O3a_1,
        clogB_VS_O3a_1,
        clogB_VS_err_O3a_1,
        clogB_TVS_O3a_1,
        clogB_TVS_err_O3a_1,
    ))
    f.write("\t\tO3b & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ \\\\\n"%(
        clogB_S_O3b_1,
        clogB_S_err_O3b_1,
        clogB_V_O3b_1,
        clogB_V_err_O3b_1,
        clogB_TS_O3b_1,
        clogB_TS_err_O3b_1,
        clogB_TV_O3b_1,
        clogB_TV_err_O3b_1,
        clogB_VS_O3b_1,
        clogB_VS_err_O3b_1,
        clogB_TVS_O3b_1,
        clogB_TVS_err_O3b_1,
    ))
    #f.write("\t\t\hline\n")
    f.write("\midrule\n")
    f.write("\t\tCombined & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ \\\\\n"%(
        clogB_S_all_1,
        clogB_S_err_all_1,
        clogB_V_all_1,
        clogB_V_err_all_1,
        clogB_TS_all_1,
        clogB_TS_err_all_1,
        clogB_TV_all_1,
        clogB_TV_err_all_1,
        clogB_VS_all_1,
        clogB_VS_err_all_1,
        clogB_TVS_all_1,
        clogB_TVS_err_all_1,
    ))
    #f.write("\t\t\hline\n")
    f.write("\\bottomrule\n")
    f.write("\t\end{tabular}\n")
    f.write("}\n")
    f.write("\caption{Combined $\log_{10}\mathcal{B}$ of the 2-detector and 3-detector events using one basis mode in the polarization test.}\n")
    f.write("\label{tab:pol_1}\n")
    f.write("\end{table}\n")

# Compute the combined log10 Bayes factor of the O2 events for the L = 2 analysis.
clogB_V_O2_2, clogB_V_err_O2_2, clogB_TS_O2_2, clogB_TS_err_O2_2, clogB_TV_O2_2, clogB_TV_err_O2_2, clogB_VS_O2_2, clogB_VS_err_O2_2, clogB_TVS_O2_2, clogB_TVS_err_O2_2 = compute_combined_logB_l_2(df_l_2_O2)

# Compute the combined log10 Bayes factor of the O3a events for the L = 2 analysis.
clogB_V_O3a_2, clogB_V_err_O3a_2, clogB_TS_O3a_2, clogB_TS_err_O3a_2, clogB_TV_O3a_2, clogB_TV_err_O3a_2, clogB_VS_O3a_2, clogB_VS_err_O3a_2, clogB_TVS_O3a_2, clogB_TVS_err_O3a_2 = compute_combined_logB_l_2(df_l_2_O3a)

# Compute the combined log10 Bayes factor of the O3b events for the L = 2 analysis.
clogB_V_O3b_2, clogB_V_err_O3b_2, clogB_TS_O3b_2, clogB_TS_err_O3b_2, clogB_TV_O3b_2, clogB_TV_err_O3b_2, clogB_VS_O3b_2, clogB_VS_err_O3b_2, clogB_TVS_O3b_2, clogB_TVS_err_O3b_2 = compute_combined_logB_l_2(df_l_2_O3b)

# Compute the combined log10 Bayes factor of all events for the L = 2 analysis.
clogB_V_all_2, clogB_V_err_all_2, clogB_TS_all_2, clogB_TS_err_all_2, clogB_TV_all_2, clogB_TV_err_all_2, clogB_VS_all_2, clogB_VS_err_all_2, clogB_TVS_all_2, clogB_TVS_err_all_2 = compute_combined_logB_l_2(df_l_2_all)

############################################################
# Write the combined results of the L = 2 analysis to file.
############################################################
with open("pol_table_2.tex", "w") as f:
    f.write("\\begin{table}\n")
    f.write("\t\\resizebox{\\textwidth}{!}{\n")
    f.write("\t\\begin{tabular}{c  c  c  c  c  c}\n")
    f.write("\t\t\\toprule\n")
    f.write("\t\tEvents & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{V}}$ & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{TS}}$ & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{TV}}$ & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{VS}}$ & $\log_{10}\mathcal{B}_{\\textrm{T}}^{\\textrm{TVS}}$\\\\\n")
    f.write("\t\t\midrule\n")
    f.write("\t\tO1 & $-$ & $-$ & $-$ & $-$ & $-$ \\\\\n")
    f.write("\t\tO2 & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ \\\\\n"%(
        clogB_V_O2_2,
        clogB_V_err_O2_2,
        clogB_TS_O2_2,
        clogB_TS_err_O2_2,
        clogB_TV_O2_2,
        clogB_TV_err_O2_2,
        clogB_VS_O2_2,
        clogB_VS_err_O2_2,
        clogB_TVS_O2_2,
        clogB_TVS_err_O2_2,
    ))
    f.write("\t\tO3a & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ \\\\\n"%(
        clogB_V_O3a_2,
        clogB_V_err_O3a_2,
        clogB_TS_O3a_2,
        clogB_TS_err_O3a_2,
        clogB_TV_O3a_2,
        clogB_TV_err_O3a_2,
        clogB_VS_O3a_2,
        clogB_VS_err_O3a_2,
        clogB_TVS_O3a_2,
        clogB_TVS_err_O3a_2,
    ))
    f.write("\t\tO3b & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ \\\\\n"%(
        clogB_V_O3b_2,
        clogB_V_err_O3b_2,
        clogB_TS_O3b_2,
        clogB_TS_err_O3b_2,
        clogB_TV_O3b_2,
        clogB_TV_err_O3b_2,
        clogB_VS_O3b_2,
        clogB_VS_err_O3b_2,
        clogB_TVS_O3b_2,
        clogB_TVS_err_O3b_2,
    ))
    f.write("\t\t\midrule\n")
    f.write("\t\tCombined & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ & $%.2f \pm %.2f$ \\\\\n"%(
        clogB_V_all_2,
        clogB_V_err_all_2,
        clogB_TS_all_2,
        clogB_TS_err_all_2,
        clogB_TV_all_2,
        clogB_TV_err_all_2,
        clogB_VS_all_2,
        clogB_VS_err_all_2,
        clogB_TVS_all_2,
        clogB_TVS_err_all_2,
    ))
    f.write("\t\t\\bottomrule\n")
    f.write("\t\end{tabular}\n")
    f.write("}\n")
    f.write("\caption{Combined $\log_{10}\mathcal{B}$ of the 3-detector events using two basis modes in the polarization test.}\n")
    f.write("\label{tab:pol_2}\n")
    f.write("\end{table}\n")

